const express = require('express');
const router = express.Router();
const jwt = require('jsonwebtoken'); // <-- ADD THIS IMPORT
const { protect, admin } = require('../middleware/auth');
const Waitlist = require('../models/Waitlist');
const Product = require('../models/Product');
const User = require('../models/User'); // Mongoose User model (already present, ensuring it's used)
const QuizQuestion = require('../models/QuizQuestion'); 
const QuizSubmission = require('../models/QuizSubmission'); // Import the model
// Utility to generate JWT token
const generateToken = (id) => {
    return jwt.sign({ id }, process.env.JWT_SECRET, {
        expiresIn: process.env.JWT_EXPIRY,
    });
};

// --- AUTH & USER MANAGEMENT ---

// @route POST /api/admin/register
// @desc Register a new admin user (Should be removed after initial setup)
// @access Public (Set up first admin)
router.post('/register', async (req, res) => {
    const { email, password } = req.body;
    try {
        const userExists = await User.findOne({ email });
        if (userExists) {
            return res.status(400).json({ message: 'User already exists' });
        }
        const user = await User.create({ email, password });
        res.status(201).json({
            _id: user._id,
            email: user.email,
            token: generateToken(user._id),
        });
    } catch (error) {
        res.status(500).json({ message: 'Registration failed' });
    }
});

// @route POST /api/admin/login
// @desc Authenticate user & get token
// @access Public
router.post('/login', async (req, res) => {
    const { email, password } = req.body;
    try {
        const user = await User.findOne({ email });
        // user.matchPassword is an asynchronous method defined on the Mongoose model
        if (user && (await user.matchPassword(password))) {
            res.json({
                _id: user._id,
                email: user.email,
                token: generateToken(user._id),
            });
        } else {
            // Changed status to 401 for authentication failure
            res.status(401).json({ message: 'Invalid email or password' });
        }
    } catch (error) {
        // Logging the error is critical for debugging 500s
        console.error('Login route error:', error); 
        res.status(500).json({ message: 'Login failed due to server error.' });
    }
});

// --- WAITLIST MANAGEMENT (PROTECTED) ---

// @route GET /api/admin/waitlist
// @desc Get all waitlist contacts
// @access Private/Admin
router.get('/waitlist', protect, async (req, res) => {
    try {
        const waitlist = await Waitlist.find().sort({ createdAt: -1 });
        res.json(waitlist);
    } catch (error) {
        res.status(500).json({ message: 'Failed to fetch waitlist' });
    }
});

// @route PUT /api/admin/waitlist/:id
// @desc Update waitlist contact status (Update existing logic for full CRUD)
// @access Private/Admin
router.put('/waitlist/:id', protect, async (req, res) => {
    const updateData = req.body; // Allows updating any field, but specifically targeting isContacted
    try {
        const updatedContact = await Waitlist.findByIdAndUpdate(
            req.params.id, 
            { $set: updateData }, 
            { new: true, runValidators: true }
        );

        if (!updatedContact) {
            return res.status(404).json({ message: 'Contact not found' });
        }
        res.json(updatedContact);
    } catch (error) {
        res.status(500).json({ message: 'Failed to update contact', error: error.message });
    }
});

// @route DELETE /api/admin/waitlist/:id
// @desc Delete a waitlist contact (NEW)
// @access Private/Admin
router.delete('/waitlist/:id', protect, async (req, res) => {
    try {
        const result = await Waitlist.findByIdAndDelete(req.params.id);
        if (!result) {
            return res.status(404).json({ message: 'Contact not found' });
        }
        res.status(204).send();
    } catch (error) {
        console.error('Error deleting waitlist contact:', error);
        res.status(500).json({ message: 'Contact deletion failed' });
    }
});

// --- PRODUCT MANAGEMENT (PROTECTED) ---

// @route GET /api/admin/products
// @desc Get all products
// @access Private/Admin
router.get('/products', async (req, res) => {
    try {
        const products = await Product.find({});
        res.json(products);
    } catch (error) {
        res.status(500).json({ message: 'Failed to fetch products' });
    }
});

// @route POST /api/admin/products
// @desc Create a new product
// @access Private/Admin
router.post('/products', protect, async (req, res) => {
    // Implementation for creating a product goes here
    const { name, price, productImageUrl, moodImageUrl, topNotes, heartNotes, baseNotes } = req.body;
    try {
        const product = new Product(req.body);
        const createdProduct = await product.save();
        res.status(201).json(createdProduct);
    } catch (error) {
        res.status(400).json({ message: 'Product creation failed', error: error.message });
    }
});

// @route PUT /api/admin/products/:id
// @desc Update a product (NEW)
// @access Private/Admin
router.put('/products/:id', protect, async (req, res) => {
    try {
        const updatedProduct = await Product.findByIdAndUpdate(
            req.params.id, 
            req.body, // Pass the entire body for update
            { new: true, runValidators: true }
        );
        if (!updatedProduct) {
            return res.status(404).json({ message: 'Product not found' });
        }
        res.json(updatedProduct);
    } catch (error) {
        console.error('Error updating product:', error);
        res.status(400).json({ message: 'Product update failed', error: error.message });
    }
});

// @route DELETE /api/admin/products/:id
// @desc Delete a product (NEW)
// @access Private/Admin
router.delete('/products/:id', protect, async (req, res) => {
    try {
        const result = await Product.findByIdAndDelete(req.params.id);
        if (!result) {
            return res.status(404).json({ message: 'Product not found' });
        }
        res.status(204).send();
    } catch (error) {
        console.error('Error deleting product:', error);
        res.status(500).json({ message: 'Product deletion failed' });
    }
});

// @route GET /api/admin/quiz/questions
// @desc Get all quiz questions (ordered)
// @access Private/Admin
router.get('/quiz/questions', async (req, res) => {
    try {
        // Find all questions and sort by the 'order' field
        const questions = await QuizQuestion.find().sort({ order: 1 });
        res.json(questions);
    } catch (error) {
        console.error('Error fetching quiz questions:', error);
        res.status(500).json({ message: 'Failed to fetch quiz questions' });
    }
});

// @route POST /api/admin/quiz/questions
// @desc Create a new quiz question
// @access Private/Admin
router.post('/quiz/questions', protect, async (req, res) => {
    let questionData = { ...req.body }; // Use let to allow modification

    // FIX: If the step type is 'input', ensure the options array is empty to bypass validation
    if (questionData.stepType === 'input') {
        questionData.options = [];
    }
    
    try {
        // Automatically determine the next order number
        const lastQuestion = await QuizQuestion.findOne().sort({ order: -1 });
        const nextOrder = lastQuestion ? lastQuestion.order + 1 : 1;
        
        questionData = {
            ...questionData,
            order: nextOrder,
        };

        const question = new QuizQuestion(questionData);
        await question.save();
        res.status(201).json(question);
    } catch (error) {
        console.error('Error creating quiz question:', error);
        res.status(400).json({ message: 'Question creation failed', error: error.message });
    }
});

// @route PUT /api/admin/quiz/questions/:id
// @desc Update a quiz question
// @access Private/Admin
router.put('/quiz/questions/:id', protect, async (req, res) => {
    let updateData = { ...req.body }; // Use let to allow modification

    // FIX: If the step type is 'input', ensure the options array is empty to bypass validation
    if (updateData.stepType === 'input') {
        updateData.options = [];
    }

    try {
        const updatedQuestion = await QuizQuestion.findByIdAndUpdate(
            req.params.id, 
            updateData, 
            { new: true, runValidators: true }
        );
        if (!updatedQuestion) {
            return res.status(404).json({ message: 'Question not found' });
        }
        res.json(updatedQuestion);
    } catch (error) {
        console.error('Error updating quiz question:', error);
        res.status(400).json({ message: 'Question update failed', error: error.message });
    }
});

// @route DELETE /api/admin/quiz/questions/:id
// @desc Delete a quiz question
// @access Private/Admin
router.delete('/quiz/questions/:id', protect, async (req, res) => {
    try {
        const result = await QuizQuestion.findByIdAndDelete(req.params.id);
        if (!result) {
            return res.status(404).json({ message: 'Question not found' });
        }
        res.status(204).send();
    } catch (error) {
        console.error('Error deleting quiz question:', error);
        res.status(500).json({ message: 'Question deletion failed' });
    }
});
// NOTE: Add routes for PUT /api/admin/products/:id and DELETE /api/admin/products/:id for full CRUD
// --- QUIZ SUBMISSION MANAGEMENT (PROTECTED) ---

// @route GET /api/admin/quiz/submissions
// @desc Get all quiz submissions (for analysis)
// @access Private/Admin
router.get('/quiz/submissions', protect, async (req, res) => {
    try {
        const submissions = await QuizSubmission.find({})
            // Populate recommendations if needed, assuming the recommendations field stores product ObjectIds
            // .populate('recommendations', 'name price') 
            .sort({ createdAt: -1 });

        res.json(submissions);
    } catch (error) {
        console.error('Failed to fetch quiz submissions:', error);
        res.status(500).json({ message: 'Failed to fetch quiz submissions for analysis.' });
    }
});

router.get('/dashboard', protect, async (req, res) => {
    try {
        // Calculate all metrics concurrently
        const [
            totalWaitlist,
            activeProducts,
            totalQuizSubmissions,
            // Mock total revenue since there is no order model yet
            // In a real app, this would query the Order model
        ] = await Promise.all([
            Waitlist.countDocuments(),
            Product.countDocuments(),
            QuizSubmission.countDocuments(),
        ]);

        const metrics = {
            totalWaitlist: totalWaitlist.toLocaleString(),
            activeProducts: activeProducts.toString(),
            // Placeholder/Mock revenue until Order model is implemented
            totalRevenue: '€85,000', 
            totalQuizSubmissions: totalQuizSubmissions.toLocaleString(),
        };

        res.json(metrics);

    } catch (error) {
        console.error('Failed to fetch dashboard metrics:', error);
        res.status(500).json({ message: 'Server error fetching dashboard metrics.' });
    }
});


module.exports = router;