const nodemailer = require('nodemailer');
const transporter = require('./email').transporter; // Reusing the main transporter

/**
 * Sends an email containing quiz recommendations to the client.
 * * @param {string} recipientEmail - The user's email address.
 * @param {string} recipientName - The user's name.
 * @param {string} scentType - The dominant result (e.g., 'fresh', 'dark').
 * @param {object[]} recommendations - Array of formatted product objects.
 */
const sendQuizRecommendationEmail = async (recipientEmail, recipientName, scentType, recommendations) => {
    
    // Format the recommendations into HTML cards for the email body
    const recommendationCards = recommendations.map(product => `
        <div style="width: 100%; max-width: 250px; margin: 0 10px 20px; text-align: center; display: inline-block; vertical-align: top;">
            <img src="${product.imageUrl}" alt="${product.name}" 
                 style="width: 100%; height: auto; border-radius: 4px; object-fit: contain; margin-bottom: 15px;">
            <h4 style="font-size: 16px; color: #182A2E; margin: 0 0 5px; text-transform: uppercase;">${product.name}</h4>
            <p style="font-size: 14px; color: #182A2E; margin: 0 0 10px;">${product.price}</p>
            <p style="font-size: 12px; color: #182A2E99; margin: 0 0 15px;">${product.description.substring(0, 80)}...</p>
            <a href="${process.env.CLIENT_URL || 'http://localhost:5173'}${product.link}" 
               style="display: inline-block; background-color: #F0CC6E; color: #182A2E; padding: 8px 15px; border-radius: 4px; text-decoration: none; font-weight: bold; font-size: 12px;">
                VIEW PRODUCT
            </a>
        </div>
    `).join('');

    const htmlContent = `
        <div style="font-family: 'Inter', sans-serif; max-width: 600px; margin: auto; border: 1px solid #E5E7EB; border-radius: 8px; overflow: hidden; background-color: #FAF7F0;">
            
            <div style="background-color: #182A2E; padding: 25px 30px; text-align: center;">
                <h1 style="color: #F0CC6E; font-family: 'Times New Roman', serif; font-size: 30px; margin: 0; letter-spacing: 2px;">MIRAGEM</h1>
                <p style="color: #FAF7F0; font-size: 14px; margin: 5px 0 0;">Your Personalized Scent Match</p>
            </div>

            <div style="padding: 30px; color: #182A2E; text-align: center;">
                <h2 style="font-size: 24px; margin-top: 0; font-weight: 400;">Hello ${recipientName},</h2>
                <p style="font-size: 16px; line-height: 1.6;">
                    Thank you for taking our quiz! We've analyzed your preferences and determined your essence is: 
                    <strong style="text-transform: uppercase; color: #F0CC6E;">${scentType}</strong>.
                </p>
                <p style="font-size: 16px; line-height: 1.6; margin-bottom: 30px;">
                    Based on your results, here are the Miragem fragrances we recommend for you:
                </p>

                <div style="text-align: center; margin-bottom: 20px;">
                    ${recommendationCards}
                </div>

                <p style="font-size: 14px; line-height: 1.6; color: #182A2E99;">
                    These recommendations capture the core of your personality and style. Click below to explore them further!
                </p>

            </div>

            <div style="background-color: #182A2E; color: #FAF7F0; padding: 15px 30px; text-align: center; font-size: 12px;">
                <p style="margin: 0;">&copy; ${new Date().getFullYear()} Miragem by Enid Nkini. The Journey Awaits.</p>
            </div>
        </div>
    `;

    try {
        const mailOptions = {
            from: process.env.SENDER_EMAIL,
            to: recipientEmail,
            subject: `Miragem Quiz Results: Your Perfect Scent Match (${scentType.toUpperCase()})`,
            html: htmlContent,
        };

        await transporter.sendMail(mailOptions);
        console.log(`Quiz recommendation email sent successfully to ${recipientEmail}`);
        return true;
    } catch (error) {
        console.error(`Error sending quiz email to ${recipientEmail}:`, error);
        return false;
    }
};

module.exports = { sendQuizRecommendationEmail };